' ----------------------------------------------------------------------------
' RegEX function to retrieve JSON values from Journal
' ----------------------------------------------------------------------------
Function RegFind:String(s:String, pattern:String, p:Int = 1)
	
	Local regex:TRegEx = TRegEx.Create(pattern)
	
	Try
	
		Local match:TRegExMatch = regex.Find(s)
		If match Then Return match.SubExp(p)
		
	Catch e:TRegExException
		
		Return Null
		
	End Try

End Function

' ----------------------------------------------------------------------------
' RegEX function to split systems into a string with a divider
' ----------------------------------------------------------------------------
Function RegFindSystem:String(s:String, pattern:String)
	
	Local match:TRegExMatch
	Local regex:TRegEx = TRegEx.Create(pattern)
	Local out:String
	
	Try

		match = regex.Find(s)
		If match Then
		
			For Local i:Int = 1 Until match.SubCount()
			
				Local suffix:String = ""
				If i > 1 Then suffix = "|"
				out:+(suffix + match.subexp(i))
	
			Next
			
		End If
		
		Return out

		Catch e:TRegExException
		
			Return "-"
			
	End Try
	
End Function

' ------------------------------------------------------------------------------------------------
' returns the exact! unix timestamp for given YEARS,MONTHS,DAYS,HOURS,MINUTES,SECONDS
' ------------------------------------------------------------------------------------------------
Function GetUnixTimestamp:Int(yy:Int, mm:Int, dd:Int, hh:Int, ii:Int, ss:Int)

	Local days:Int[] = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334]
 	Local years:Int = yy - 1970
	Local leap:Int = ((yy - 1) - 1968) / 4 - ((yy - 1) - 1900) / 100 + ((yy - 1) - 1600) / 400
	
	Local unixtime:Int = ss + 60 * ii + 60 * 60 * hh + (days[mm - 1] + dd - 1) * 60 * 60 * 24 + (years * 365 + leap) * 60 * 60 * 24
 
  	If ((mm > 2) And (yy Mod 4 = 0 And (yy Mod 100 <> 0 Or yy Mod 400 = 0))) Then
	
    	' Schalttag
		unixtime:+(60 * 60 * 24)
		
	End If
 
  Return unixtime + (timecorrection * 3600)

End Function 

' ------------------------------------------------------------------------------------------------
' Return formatted unix timestamp
' ------------------------------------------------------------------------------------------------
' %Y = 1970 (YYYY)
' %m = 09   (MM)
' %d = 01   (DD)
' %H = 19   (HH)
' %M = 23   (II)
' %S = 11   (SS)
' ------------------------------------------------------------------------------------------------
Function FTimestamp:String(timestamp:Int, format:String = "%Y-%m-%d %H:%M:%S")

	Local time:Int Ptr, buff:Byte[256]
	
	time = VarPtr(timestamp)
	strftime_(buff, 256, format, localtime_(time))
	
	Return String.FromCString(buff)
	
End Function

' ------------------------------------------------------------------------------------------------
' Check if an INT number is within a range or returns a given default value
' ------------------------------------------------------------------------------------------------
Function CheckINT:Int(Val:Int, vmin:Int, vmax:Int, def:Int)

	'If (Val = Null) Then Return def

	If Val >= vmin And Val <= vmax Then Return Val Else Return def

End Function

' ------------------------------------------------------------------------------------------------
' INT round in both directions
' ------------------------------------------------------------------------------------------------
Function RoundInt:Int(x:Float)
	
	If x >= 0 Then Return Floor(x) Else Return Ceil(x)
	
End Function

' --------------------------------------------------------------------------------------------
' get current UNIX timestamp from OS
' --------------------------------------------------------------------------------------------
Function GetTimeStamp:Int()

	Local time:Int[256]
	time_(time)
	return time[0] + (timecorrection * 3600)

End Function


' --------------------------------------------------------------------------------------------
' get current UNIX timestamp from OS
' --------------------------------------------------------------------------------------------
Function GetTimeStampLocal:Int()

	Local time:Int[256]
	time_(time)
	Return time[0]

End Function

' ----------------------------------------------------------------------------
' Get current date
' ----------------------------------------------------------------------------
Function CurrentUnixTimestamp:String(format:String = "%Y-%m-%d %H:%M:%S")

	Local time:Int[256]
	Local buff:Byte[256]

	'fill buffer
	time_(time)
	
	'format buffer
	strftime_(buff,256,format,localtime_( time ))

	Return String.FromCString(buff)
	
End Function

' ------------------------------------------------------------------------------------------------
' Output an int with dividing dots
' ------------------------------------------------------------------------------------------------
Function FormatINT:String(Value:String, div:String = ".")

	Local i:Int
	Local out:String
	Local Length:Int = Len(Value)

	For i = Length - 1 To 0 Step - 1

		out = Mid(Value, 1 + i, 1) + out
		If i And Not ((Length - i) Mod 3) Then out = div + out
		
	Next

	Return out

End Function

' ------------------------------------------------------------------------------------------------
' Return a nicely formatted number
' ------------------------------------------------------------------------------------------------
Function FormatMath:String(Value:Double, floats:Int = 2, delimiter:Int = True)

	Local s:String=value
	Local e:Int=s.Find("e")
	Local exponent:Int

	Local i:Int=s.Find(".")

	Local integer:String
	Local floating:String

	' find out 10^exponent, positive and negative
	If e>0 Then exponent=Int(Mid(s,e+2,999)) Else exponent=Null

	' find out integer
	If i>0 Then

		integer=Mid(s,0,i+1)
		floating=Mid(s,i+2,floats)
		
		'If floats=0 Or Int(floating)=0 Then floating=Null
		If floats=0 Then floating=Null

	Else

		integer=s
		floating=Null
		
	EndIf

	' add dot dividers for integer
	If delimiter Then

		Local l:Int=Len(integer)
		Local out:String
		Local negative:Int

		If integer<0 Then negative=1

		For Local j:Int=l To 0 Step -1

			out:+Mid(integer, l - j, 1)

			If (j Mod 3) = 0 And j > 0 And (j < l - negative) Then out:+"."

		Next

		integer=out

	EndIf
	
	' output with exponent
	If floating And exponent And floating Then Return integer+","+floating+"x10^"+exponent
	If integer And exponent And (Not floating) Then Return integer+"x10^"+exponent
	
	' output without exponent
	If floating And (Not exponent) Then Return integer + "," + floating
	If (Not floating) And (Not exponent) Then Return integer

EndFunction

' ------------------------------------------------------------------------------------------------
' Debug Gadget output
' ------------------------------------------------------------------------------------------------
Function LogUpdate(txt:String, statusbar:Int = False)
	
	If Debug Then AddTextAreaText(LogDebug, txt + "~n")
	
	If statusbar Then SetStatusText(Window, txt)
	
	RedrawGadget(LogDebug)
		
End Function

' ----------------------------------------------------------------------------
' help function to add menu entries
' ----------------------------------------------------------------------------
Function AddMenu:TGadget(title:String, menu:TGadget, constant:Int, icon:Int, iconstrip:TIconStrip, shortcut:Int, modifier:Int)

	Local gadget:TGadget = CreateMenu(title, constant, menu, shortcut, modifier)

	SetGadgetPixmap(gadget, PixmapFromIconStrip(iconstrip, icon), GADGETPIXMAP_ICON | GADGETPIXMAP_NOTEXT)
	
	Return gadget

End Function

' ----------------------------------------------------------------------------
' create a new document within a tab
' ----------------------------------------------------------------------------
Function CreateDocument:TGadget(tabber:TGadget)

	Local panel:TGadget
	panel = CreatePanel(0, 0, ClientWidth(tabber), ClientHeight(tabber), tabber)
	SetGadgetLayout panel,1,1,1,1
	HideGadget panel
	
	Return panel
	
End Function

' --------------------------------------------------------------------------------------------
' Adds additional characters to a string to make all lines equal in width
' --------------------------------------------------------------------------------------------
Function FillSpaces:String(value:String, Length:Int = 26, character:String = " ")

	Local l:Int = Len(Value)
	Local s:String
	
	If Length > (l - 1) Then
	
		For Local i:Int = 0 To Length - l - 1
	
			s:+character

		Next
		
	EndIf
	
	Return Value + s
	
End Function

' --------------------------------------------------------------------------------------------
' Trim leading space (left part of string)
' --------------------------------------------------------------------------------------------
Function LTrim:String(str:String)

	If Left(str, 1) = " " Then Return Mid(str, 2, str.Length - 1) Else Return str

End Function

' ------------------------------------------------------------------------------------------------
' Draw a OpenGL Rectangle
' ------------------------------------------------------------------------------------------------
Function BeginLines()

	' Begin drawin GL Lines
	glDisable(GL_TEXTURE_2D)
	glEnable(GL_COLOR_MATERIAL)
	glBegin(GL_LINES)
	glLineWidth(1.0)
	
End Function

' ------------------------------------------------------------------------------------------------
' Draw a OpenGL Rectangle
' ------------------------------------------------------------------------------------------------
Function EndLines()

	' End drawing GL Lines
	glEnd()
	glDisable(GL_COLOR_MATERIAL)
	glEnable(GL_TEXTURE_2D)

End Function

' ------------------------------------------------------------------------------------------------
' Draw a OpenGL Rectangle
' ------------------------------------------------------------------------------------------------
Function DrawGRect(x:Int, y:Int, w:Int, h:Int)

	DrawGLine(x-1, y, x + w, y)
	DrawGLine(x + w, y, x + w, y + h)
	DrawGLine(x, y + h, x + w, y + h)
	DrawGLine(x, y, x, y + h)

End Function

' ------------------------------------------------------------------------------------------------
' Draw a OpenGL Line
' ------------------------------------------------------------------------------------------------
Function DrawGLine(x1:Int, y1:Int, x2:Int, y2:Int)

	glVertex2i(x1, y1)
	glVertex2i(x2, y2)
	
End Function

Function FormatDigit:String(val:Int)

	If val < 10 Then Return "0" + val
	
	Return val

End Function